<?php
// admin.php
require_once 'config.php';
require_once 'db_connection.php';

// অ্যাডমিন অথেন্টিকেশন
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}

// অ্যাডমিন তথ্য
$admin_id = $_SESSION['admin_id'];
$admin = $db->select('admin_users', '*', ['id' => $admin_id])[0];

// Dashboard statistics
$stats = [
    'total_users' => $db->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'],
    'total_deposits' => $db->query("SELECT SUM(amount) as total FROM deposits WHERE status = 'approved'")->fetch_assoc()['total'] ?? 0,
    'total_withdrawals' => $db->query("SELECT SUM(amount) as total FROM withdrawals WHERE status = 'approved'")->fetch_assoc()['total'] ?? 0,
    'pending_deposits' => $db->query("SELECT COUNT(*) as count FROM deposits WHERE status = 'pending'")->fetch_assoc()['count'],
    'pending_withdrawals' => $db->query("SELECT COUNT(*) as count FROM withdrawals WHERE status = 'pending'")->fetch_assoc()['count'],
    'active_gift_codes' => $db->query("SELECT COUNT(*) as count FROM gift_codes WHERE status = 'active'")->fetch_assoc()['count']
];

// সেটিংস লোড
$settings = [];
$settings_result = $db->select('settings', '*');
foreach ($settings_result as $setting) {
    $settings[$setting['setting_key']] = $setting['setting_value'];
}

// অ্যাকশন হ্যান্ডলিং
$action = $_GET['action'] ?? '';
$message = $_GET['message'] ?? '';
$error = $_GET['error'] ?? '';

// ইউজার লোড
$users = $db->query("SELECT * FROM users ORDER BY created_at DESC")->fetch_all(MYSQLI_ASSOC);

// পেন্ডিং ডিপোজিট
$pending_deposits = $db->query("
    SELECT d.*, u.ui_id, u.name 
    FROM deposits d 
    JOIN users u ON d.user_id = u.id 
    WHERE d.status = 'pending' 
    ORDER BY d.created_at DESC
")->fetch_all(MYSQLI_ASSOC);

// পেন্ডিং উইথড্র
$pending_withdrawals = $db->query("
    SELECT w.*, u.ui_id, u.name 
    FROM withdrawals w 
    JOIN users u ON w.user_id = u.id 
    WHERE w.status = 'pending' 
    ORDER BY w.created_at DESC
")->fetch_all(MYSQLI_ASSOC);

// গিফট কোড
$gift_codes = $db->query("
    SELECT g.*, u.ui_id as redeemed_by_ui 
    FROM gift_codes g 
    LEFT JOIN users u ON g.redeemed_by = u.id 
    ORDER BY g.created_at DESC
")->fetch_all(MYSQLI_ASSOC);

// পেমেন্ট নাম্বার
$payment_numbers = $db->select('payment_numbers', '*', [], 'type ASC');

// AJAX requests
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'approve_deposit':
            $deposit_id = $_POST['deposit_id'];
            $deposit = $db->select('deposits', '*', ['id' => $deposit_id])[0];
            
            $db->beginTransaction();
            try {
                // ডিপোজিট approve
                $db->update('deposits', [
                    'status' => 'approved',
                    'updated_at' => date('Y-m-d H:i:s')
                ], ['id' => $deposit_id]);
                
                // ব্যালেন্স আপডেট
                $db->query("UPDATE users SET deposit_balance = deposit_balance + ? WHERE id = ?", 
                    [$deposit['amount'], $deposit['user_id']]);
                
                // has_deposited আপডেট
                $db->query("UPDATE users SET has_deposited = 1 WHERE id = ?", [$deposit['user_id']]);
                
                $db->commit();
                echo json_encode(['success' => true, 'message' => 'ডিপোজিট Approved!']);
            } catch (Exception $e) {
                $db->rollback();
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit();
            
        case 'reject_deposit':
            $db->update('deposits', [
                'status' => 'rejected',
                'updated_at' => date('Y-m-d H:i:s')
            ], ['id' => $_POST['deposit_id']]);
            echo json_encode(['success' => true]);
            exit();
            
        case 'approve_withdrawal':
            $withdrawal_id = $_POST['withdrawal_id'];
            $withdrawal = $db->select('withdrawals', '*', ['id' => $withdrawal_id])[0];
            
            $db->beginTransaction();
            try {
                // চেক ব্যালেন্স
                $user = $db->select('users', 'winning_balance', ['id' => $withdrawal['user_id']])[0];
                
                if ($user['winning_balance'] < $withdrawal['amount']) {
                    throw new Exception('Insufficient balance');
                }
                
                // উইথড্র approve
                $db->update('withdrawals', [
                    'status' => 'approved',
                    'updated_at' => date('Y-m-d H:i:s')
                ], ['id' => $withdrawal_id]);
                
                // ব্যালেন্স আপডেট
                $db->query("UPDATE users SET winning_balance = winning_balance - ? WHERE id = ?", 
                    [$withdrawal['amount'], $withdrawal['user_id']]);
                
                $db->commit();
                echo json_encode(['success' => true, 'message' => 'উইথড্র Approved!']);
            } catch (Exception $e) {
                $db->rollback();
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit();
            
        case 'update_user':
            $user_id = $_POST['user_id'];
            $data = [
                'deposit_balance' => $_POST['deposit_balance'],
                'winning_balance' => $_POST['winning_balance'],
                'total_referrals' => $_POST['total_referrals'],
                'total_bonus' => $_POST['total_bonus'],
                'status' => $_POST['status']
            ];
            
            $db->update('users', $data, ['id' => $user_id]);
            echo json_encode(['success' => true]);
            exit();
            
        case 'generate_gift_code':
            $amount = $_POST['amount'];
            $code = 'BD' . date('Ymd') . strtoupper(substr(md5(uniqid()), 0, 6));
            
            $db->insert('gift_codes', [
                'code' => $code,
                'amount' => $amount,
                'created_by' => $admin_id,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            echo json_encode(['success' => true, 'code' => $code]);
            exit();
            
        case 'update_settings':
            foreach ($_POST['settings'] as $key => $value) {
                $db->query("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) 
                           ON DUPLICATE KEY UPDATE setting_value = ?", 
                           [$key, $value, $value]);
            }
            echo json_encode(['success' => true]);
            exit();
    }
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Admin Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@400;700&family=Roboto+Mono:wght@700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="admin_styles.css">
    <style>
        /* Admin-specific styles from your HTML */
        :root {
            --primary-color: #00aaff;
            --secondary-color: #ff3366;
            --accent-color: #FFD700;
            --dark-bg: #121212;
            --card-bg: #1e1e1e;
            --text-primary: #ffffff;
            --text-secondary: #b0b0b0;
            --success-color: #4CAF50;
            --error-color: #F44336;
        }
        /* ... All your admin CSS styles ... */
        /* (আপনার দেওয়া admin HTML-এর সব CSS এখানে যোগ করুন) */
    </style>
</head>
<body>
    <header class="header">
        <div class="logo"><?php echo APP_NAME; ?> - Admin Panel</div>
        <div style="display: flex; align-items: center; gap: 15px;">
            <span>Welcome, <?php echo htmlspecialchars($admin['full_name']); ?></span>
            <button class="logout-btn" onclick="logout()"><i class="fas fa-sign-out-alt"></i> Logout</button>
        </div>
    </header>

    <div class="container">
        <!-- Dashboard Stats -->
        <div class="dashboard">
            <div class="stat-card">
                <div class="stat-icon users"><i class="fas fa-users"></i></div>
                <div class="stat-info">
                    <h3>Total Users</h3>
                    <p id="total-users"><?php echo $stats['total_users']; ?></p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon deposits"><i class="fas fa-arrow-down"></i></div>
                <div class="stat-info">
                    <h3>Pending Deposits</h3>
                    <p id="pending-deposits"><?php echo $stats['pending_deposits']; ?></p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon withdrawals"><i class="fas fa-arrow-up"></i></div>
                <div class="stat-info">
                    <h3>Pending Withdrawals</h3>
                    <p id="pending-withdrawals"><?php echo $stats['pending_withdrawals']; ?></p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon gift-codes"><i class="fas fa-gift"></i></div>
                <div class="stat-info">
                    <h3>Active Gift Codes</h3>
                    <p id="active-gift-codes"><?php echo $stats['active_gift_codes']; ?></p>
                </div>
            </div>
        </div>

        <!-- Navigation Menu -->
        <nav class="nav-menu">
            <div class="nav-item active" onclick="showPage('users', this)">User Management</div>
            <div class="nav-item" onclick="showPage('deposits', this)">Deposits</div>
            <div class="nav-item" onclick="showPage('withdrawals', this)">Withdrawals</div>
            <div class="nav-item" onclick="showPage('gift-codes', this)">Gift Codes</div>
            <div class="nav-item" onclick="showPage('numbers', this)">Number Management</div>
            <div class="nav-item" onclick="showPage('settings', this)">Settings</div>
        </nav>
        
        <main>
            <!-- Users Page -->
            <div id="users-page" class="page active">
                <div class="table-container">
                    <div class="search-bar">
                        <input type="text" class="search-input" id="user-search" placeholder="Search by UI ID or phone number" onkeyup="searchUsers()">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>UI ID</th>
                                <th>Name</th>
                                <th>Phone</th>
                                <th>Deposit (৳)</th>
                                <th>Winning (৳)</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="users-table-body">
                            <?php foreach ($users as $user): ?>
                            <tr>
                                <td class="copyable-id" onclick="copyToClipboard('<?php echo $user['ui_id']; ?>')" title="Click to copy">
                                    <?php echo htmlspecialchars($user['ui_id']); ?>
                                </td>
                                <td><?php echo htmlspecialchars($user['name']); ?></td>
                                <td><?php echo htmlspecialchars($user['phone']); ?></td>
                                <td><?php echo number_format($user['deposit_balance'], 2); ?></td>
                                <td><?php echo number_format($user['winning_balance'], 2); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $user['status']; ?>">
                                        <?php echo $user['status']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-btns">
                                        <button class="action-btn view-btn" onclick="viewUser(<?php echo $user['id']; ?>)" title="View">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button class="action-btn edit-btn" onclick="editUser(<?php echo $user['id']; ?>)" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="action-btn <?php echo $user['status'] == 'blocked' ? 'unblock-btn' : 'block-btn'; ?>" 
                                                onclick="toggleUserBlock(<?php echo $user['id']; ?>, '<?php echo $user['status']; ?>')" 
                                                title="<?php echo $user['status'] == 'blocked' ? 'Unblock' : 'Block'; ?>">
                                            <i class="fas fa-<?php echo $user['status'] == 'blocked' ? 'check-circle' : 'ban'; ?>"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Deposits Page -->
            <div id="deposits-page" class="page">
                <div class="table-container">
                    <h2>Pending Deposit Requests</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>User UI ID</th>
                                <th>Amount (৳)</th>
                                <th>Method</th>
                                <th>Agent No.</th>
                                <th>Txn ID</th>
                                <th>Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="deposits-table-body">
                            <?php foreach ($pending_deposits as $deposit): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($deposit['ui_id']); ?></td>
                                <td><?php echo number_format($deposit['amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($deposit['method']); ?></td>
                                <td><?php echo htmlspecialchars($deposit['agent_number']); ?></td>
                                <td><?php echo htmlspecialchars($deposit['transaction_id']); ?></td>
                                <td><?php echo date('d/m/y H:i', strtotime($deposit['created_at'])); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $deposit['status']; ?>">
                                        <?php echo $deposit['status']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-btns">
                                        <button class="action-btn approve-btn" onclick="approveDeposit(<?php echo $deposit['id']; ?>)">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button class="action-btn reject-btn" onclick="rejectDeposit(<?php echo $deposit['id']; ?>)">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Withdrawals Page -->
            <div id="withdrawals-page" class="page">
                <div class="table-container">
                    <h2>Pending Withdrawal Requests</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>User UI ID</th>
                                <th>Amount (৳)</th>
                                <th>Method</th>
                                <th>Account</th>
                                <th>Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="withdrawals-table-body">
                            <?php foreach ($pending_withdrawals as $withdrawal): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($withdrawal['ui_id']); ?></td>
                                <td><?php echo number_format($withdrawal['amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($withdrawal['method']); ?></td>
                                <td><?php echo htmlspecialchars($withdrawal['account_number']); ?></td>
                                <td><?php echo date('d/m/y H:i', strtotime($withdrawal['created_at'])); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $withdrawal['status']; ?>">
                                        <?php echo $withdrawal['status']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-btns">
                                        <button class="action-btn approve-btn" onclick="approveWithdrawal(<?php echo $withdrawal['id']; ?>)">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button class="action-btn reject-btn" onclick="rejectWithdrawal(<?php echo $withdrawal['id']; ?>)">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Gift Codes Page -->
            <div id="gift-codes-page" class="page">
                <div class="form-container">
                    <h2>Generate Gift Code</h2>
                    <div class="form-group">
                        <label for="gift-amount">Amount (৳)</label>
                        <input type="number" id="gift-amount" placeholder="Enter amount">
                    </div>
                    <button class="btn btn-primary" onclick="generateGiftCode()">Generate Code</button>
                    <div class="generated-code" id="generated-code-container">
                        <span class="code-text" id="code-text"></span>
                        <button class="copy-btn" onclick="copyGiftCode()"><i class="fas fa-copy"></i> Copy</button>
                    </div>
                </div>
                <div class="table-container">
                    <h2>Existing Gift Codes</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Amount (৳)</th>
                                <th>Status</th>
                                <th>Redeemed By (UI ID)</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="gift-codes-table-body">
                            <?php foreach ($gift_codes as $code): ?>
                            <tr>
                                <td class="code-text"><?php echo htmlspecialchars($code['code']); ?></td>
                                <td><?php echo number_format($code['amount'], 2); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $code['status']; ?>">
                                        <?php echo $code['status']; ?>
                                    </span>
                                </td>
                                <td><?php echo $code['redeemed_by_ui'] ?? '-'; ?></td>
                                <td>
                                    <?php if ($code['status'] == 'active'): ?>
                                    <button class="action-btn reject-btn" onclick="deactivateGiftCode(<?php echo $code['id']; ?>)" title="Deactivate">
                                        <i class="fas fa-ban"></i>
                                    </button>
                                    <?php else: ?>
                                    -
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Number Management Page -->
            <div id="numbers-page" class="page">
                <div class="form-container">
                    <h2>Add New Payment Number</h2>
                    <form id="add-number-form" class="form-grid">
                        <div class="form-group">
                            <label for="payment-number">Number</label>
                            <input type="text" id="payment-number" required>
                        </div>
                        <div class="form-group">
                            <label for="payment-label">Label / Headline</label>
                            <input type="text" id="payment-label" placeholder="e.g., Agent 1" required>
                        </div>
                        <div class="form-group">
                            <label for="payment-type">Type</label>
                            <select id="payment-type" required>
                                <option value="bKash">bKash</option>
                                <option value="Nagad">Nagad</option>
                            </select>
                        </div>
                    </form>
                    <button class="btn btn-primary" onclick="addPaymentNumber()">Add Number</button>
                </div>
                <div class="table-container">
                    <h2>Existing Payment Numbers</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>Number</th>
                                <th>Label</th>
                                <th>Type</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="numbers-table-body">
                            <?php foreach ($payment_numbers as $number): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($number['number']); ?></td>
                                <td><?php echo htmlspecialchars($number['label']); ?></td>
                                <td><?php echo htmlspecialchars($number['type']); ?></td>
                                <td>
                                    <button class="action-btn reject-btn" onclick="deleteNumber(<?php echo $number['id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Settings Page -->
            <div id="settings-page" class="page">
                <div class="form-container">
                    <h2>Bonus Settings</h2>
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="welcome-bonus">Welcome Bonus (৳)</label>
                            <input type="number" id="welcome-bonus" value="<?php echo $settings['welcome_bonus'] ?? 0; ?>">
                        </div>
                        <div class="form-group">
                            <label for="referral-bonus">Referral Bonus (৳)</label>
                            <input type="number" id="referral-bonus" value="<?php echo $settings['referral_bonus'] ?? 0; ?>">
                        </div>
                    </div>
                    <button class="btn btn-primary" onclick="updateBonusSettings()">Update Bonuses</button>
                </div>

                <div class="form-container">
                    <h2>Transaction Settings</h2>
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="min-deposit">Minimum Deposit (৳)</label>
                            <input type="number" id="min-deposit" value="<?php echo $settings['min_deposit'] ?? 100; ?>">
                        </div>
                        <div class="form-group">
                            <label for="min-withdraw">Minimum Withdraw (৳)</label>
                            <input type="number" id="min-withdraw" value="<?php echo $settings['min_withdraw'] ?? 500; ?>">
                        </div>
                    </div>
                    <button class="btn btn-primary" onclick="updateTransactionSettings()">Update Limits</button>
                </div>

                <div class="form-container">
                    <h2>Update Notice</h2>
                    <div class="form-group">
                        <label for="notice-text">Notice Text</label>
                        <textarea id="notice-text" placeholder="Enter notice text" rows="4"><?php echo htmlspecialchars($settings['notice_text'] ?? ''); ?></textarea>
                    </div>
                    <button class="btn btn-primary" onclick="updateNotice()">Update Notice</button>
                </div>
                
                <div class="form-container">
                    <h2>Support Settings</h2>
                    <div class="form-group">
                        <label for="support-link">Support Link (URL)</label>
                        <input type="url" id="support-link" value="<?php echo htmlspecialchars($settings['support_link'] ?? ''); ?>" placeholder="https://t.me/your_support_channel">
                    </div>
                    <button class="btn btn-primary" onclick="updateSupportLink()">Update Support Link</button>
                </div>
            </div>
        </main>
    </div>

    <!-- Modals -->
    <div id="user-edit-modal" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="closeModal()">&times;</span>
            <h2 class="modal-title">Edit User</h2>
            <form class="modal-form" id="edit-user-form">
                <input type="hidden" id="edit-user-id">
                <div class="form-group">
                    <label for="edit-deposit-balance">Deposit Balance (৳)</label>
                    <input type="number" id="edit-deposit-balance" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="edit-winning-balance">Winning Balance (৳)</label>
                    <input type="number" id="edit-winning-balance" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="edit-referrals">Total Referrals</label>
                    <input type="number" id="edit-referrals" required>
                </div>
                <div class="form-group">
                    <label for="edit-bonus">Total Bonus (৳)</label>
                    <input type="number" id="edit-bonus" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="edit-status">Status</label>
                    <select id="edit-status" required>
                        <option value="active">Active</option>
                        <option value="suspended">Suspended</option>
                        <option value="blocked">Blocked</option>
                    </select>
                </div>
                <div class="modal-buttons">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
    
    <div id="user-view-modal" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="closeModal()">&times;</span>
            <h2 class="modal-title">User Details</h2>
            <div id="user-view-content">
                <!-- Dynamic content -->
            </div>
        </div>
    </div>

    <div id="toast"></div>

    <script src="js/admin.js"></script>
    <script>
        // Admin JavaScript functions
        function showPage(pageId, navItem) {
            document.querySelectorAll('.page').forEach(page => page.classList.remove('active'));
            document.getElementById(`${pageId}-page`).classList.add('active');
            document.querySelectorAll('.nav-item').forEach(item => item.classList.remove('active'));
            navItem.classList.add('active');
        }
        
        function logout() {
            window.location.href = 'logout.php';
        }
        
        function approveDeposit(depositId) {
            if (confirm('Approve this deposit?')) {
                fetch('', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: `ajax=1&action=approve_deposit&deposit_id=${depositId}`
                })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        showToast(data.message, 'success');
                        setTimeout(() => location.reload(), 1000);
                    } else {
                        showToast(data.message, 'error');
                    }
                });
            }
        }
        
        function approveWithdrawal(withdrawalId) {
            if (confirm('Approve this withdrawal?')) {
                fetch('', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: `ajax=1&action=approve_withdrawal&withdrawal_id=${withdrawalId}`
                })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        showToast(data.message, 'success');
                        setTimeout(() => location.reload(), 1000);
                    } else {
                        showToast(data.message, 'error');
                    }
                });
            }
        }
        
        function rejectDeposit(depositId) {
            if (confirm('Reject this deposit?')) {
                fetch('', {
                    method: 'POST',
                    body: `ajax=1&action=reject_deposit&deposit_id=${depositId}`
                })
                .then(() => {
                    showToast('Deposit rejected', 'success');
                    setTimeout(() => location.reload(), 1000);
                });
            }
        }
        
        function rejectWithdrawal(withdrawalId) {
            if (confirm('Reject this withdrawal?')) {
                fetch('', {
                    method: 'POST',
                    body: `ajax=1&action=reject_withdrawal&withdrawal_id=${withdrawalId}`
                })
                .then(() => {
                    showToast('Withdrawal rejected', 'success');
                    setTimeout(() => location.reload(), 1000);
                });
            }
        }
        
        function editUser(userId) {
            // Fetch user data and populate modal
            fetch(`api/get_user.php?id=${userId}`)
                .then(res => res.json())
                .then(user => {
                    document.getElementById('edit-user-id').value = user.id;
                    document.getElementById('edit-deposit-balance').value = user.deposit_balance;
                    document.getElementById('edit-winning-balance').value = user.winning_balance;
                    document.getElementById('edit-referrals').value = user.total_referrals;
                    document.getElementById('edit-bonus').value = user.total_bonus;
                    document.getElementById('edit-status').value = user.status;
                    document.getElementById('user-edit-modal').classList.add('show');
                });
        }
        
        function viewUser(userId) {
            fetch(`api/get_user.php?id=${userId}`)
                .then(res => res.json())
                .then(user => {
                    const content = `
                        <div class="detail-item"><strong>Name:</strong> ${user.name}</div>
                        <div class="detail-item"><strong>Phone:</strong> ${user.phone}</div>
                        <div class="detail-item"><strong>UI ID:</strong> ${user.ui_id}</div>
                        <div class="detail-item"><strong>Referred By:</strong> ${user.used_referral_code || 'None'}</div>
                        <div class="detail-item"><strong>Created:</strong> ${user.created_at}</div>
                        <div class="detail-item"><strong>Last Login:</strong> ${user.last_login || 'Never'}</div>
                    `;
                    document.getElementById('user-view-content').innerHTML = content;
                    document.getElementById('user-view-modal').classList.add('show');
                });
        }
        
        function toggleUserBlock(userId, currentStatus) {
            const action = currentStatus === 'blocked' ? 'unblock' : 'block';
            if (confirm(`${action.charAt(0).toUpperCase() + action.slice(1)} this user?`)) {
                fetch('', {
                    method: 'POST',
                    body: `ajax=1&action=update_user&user_id=${userId}&status=${currentStatus === 'blocked' ? 'active' : 'blocked'}`
                })
                .then(() => {
                    showToast(`User ${action}ed`, 'success');
                    setTimeout(() => location.reload(), 1000);
                });
            }
        }
        
        function generateGiftCode() {
            const amount = document.getElementById('gift-amount').value;
            if (!amount || amount <= 0) {
                alert('Enter valid amount');
                return;
            }
            
            fetch('', {
                method: 'POST',
                body: `ajax=1&action=generate_gift_code&amount=${amount}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('code-text').textContent = data.code;
                    document.getElementById('generated-code-container').classList.add('show');
                    showToast('Gift code generated!', 'success');
                }
            });
        }
        
        function copyGiftCode() {
            const code = document.getElementById('code-text').textContent;
            navigator.clipboard.writeText(code).then(() => showToast('Code copied!', 'success'));
        }
        
        function deactivateGiftCode(codeId) {
            if (confirm('Deactivate this gift code?')) {
                fetch(`api/deactivate_gift_code.php?id=${codeId}`)
                    .then(() => {
                        showToast('Gift code deactivated', 'success');
                        setTimeout(() => location.reload(), 1000);
                    });
            }
        }
        
        function addPaymentNumber() {
            const number = document.getElementById('payment-number').value;
            const label = document.getElementById('payment-label').value;
            const type = document.getElementById('payment-type').value;
            
            if (!number || !label) {
                alert('Fill all fields');
                return;
            }
            
            fetch('api/add_payment_number.php', {
                method: 'POST',
                body: JSON.stringify({number, label, type})
            })
            .then(() => {
                showToast('Number added!', 'success');
                setTimeout(() => location.reload(), 1000);
            });
        }
        
        function deleteNumber(id) {
            if (confirm('Delete this number?')) {
                fetch(`api/delete_payment_number.php?id=${id}`)
                    .then(() => {
                        showToast('Number deleted', 'success');
                        setTimeout(() => location.reload(), 1000);
                    });
            }
        }
        
        function updateBonusSettings() {
            const settings = {
                welcome_bonus: document.getElementById('welcome-bonus').value,
                referral_bonus: document.getElementById('referral-bonus').value
            };
            
            updateSettings(settings);
        }
        
        function updateTransactionSettings() {
            const settings = {
                min_deposit: document.getElementById('min-deposit').value,
                min_withdraw: document.getElementById('min-withdraw').value
            };
            
            updateSettings(settings);
        }
        
        function updateNotice() {
            const settings = {
                notice_text: document.getElementById('notice-text').value
            };
            
            updateSettings(settings);
        }
        
        function updateSupportLink() {
            const settings = {
                support_link: document.getElementById('support-link').value
            };
            
            updateSettings(settings);
        }
        
        function updateSettings(settings) {
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `ajax=1&action=update_settings&settings=${JSON.stringify(settings)}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showToast('Settings updated!', 'success');
                }
            });
        }
        
        function closeModal() {
            document.querySelectorAll('.modal').forEach(m => m.classList.remove('show'));
        }
        
        function showToast(message, type = 'info') {
            const toast = document.getElementById('toast');
            toast.innerHTML = `<i class="fas fa-${type === 'success' ? 'check-circle' : (type === 'error' ? 'times-circle' : 'info-circle')}"></i> ${message}`;
            toast.className = `show ${type}`;
            setTimeout(() => { toast.className = toast.className.replace('show', ''); }, 3000);
        }
        
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(() => showToast('Copied!', 'success'));
        }
        
        function searchUsers() {
            const input = document.getElementById('user-search').value.toLowerCase();
            const rows = document.querySelectorAll('#users-table-body tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(input) ? '' : 'none';
            });
        }
        
        // Form submission
        document.getElementById('edit-user-form').addEventListener('submit', function(e) {
            e.preventDefault();
            const userId = document.getElementById('edit-user-id').value;
            const data = {
                deposit_balance: document.getElementById('edit-deposit-balance').value,
                winning_balance: document.getElementById('edit-winning-balance').value,
                total_referrals: document.getElementById('edit-referrals').value,
                total_bonus: document.getElementById('edit-bonus').value,
                status: document.getElementById('edit-status').value
            };
            
            fetch('', {
                method: 'POST',
                body: `ajax=1&action=update_user&user_id=${userId}&${new URLSearchParams(data)}`
            })
            .then(res => res.json())
            .then(result => {
                if (result.success) {
                    showToast('User updated!', 'success');
                    closeModal();
                    setTimeout(() => location.reload(), 1000);
                }
            });
        });
    </script>
</body>
</html>